	function gfirvhdl(dir, name, bitspec, stimlen, genopts)
	%gfirvhdl(dir, name, bitspec, stimlen, genopts)
	%
	%Generates VHDL code for reduction tree and environment.
	%
	%Generated parts:
	%  VHDL code for bit-level implementation (name.vhdl)
	%  VHDL code for FIR DF reference filter (nameref.vhdl)
	%  VHDL testbench (name_tb.vhdl)
	%  Stimuli files for testbench (x_%d.stim, y_%d.stim)
	%  Script for compiling and simulating using Mentor Modelsim
	%
	%Arguments:
	%  dir - directory to generate files in
	%  name - name of generated filter
	%  bitspec - specification structure
	%  stimlen - stimuli length in samples
	%  genopts - generator options
	
	%Copyright 2008, 2010 Anton Blad
	%
	%This file is part of firgen.
	%
	%firgen is free software: you can redistribute it and/or modify
	%it under the terms of the GNU General Public License as published by
	%the Free Software Foundation, either version 3 of the License, or
	%(at your option) any later version.
	%
	%firgen is distributed in the hope that it will be useful,
	%but WITHOUT ANY WARRANTY; without even the implied warranty of
	%MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	%GNU General Public License for more details.
	%
	%You should have received a copy of the GNU General Public License
	%along with firgen.  If not, see <http://www.gnu.org/licenses/>
	
	if nargin < 5
		genopts = [];
	end
	genopts = parseopts(genopts);
	
	if ~isdir(dir)
		mkdir(dir);
	end
	
	if genopts.verbose >= 1
		disp(sprintf('Generating VHDL code in %s', dir));
	end
	
	% Generate VHDL code for full adders, half adders, and registers
	if genopts.verbose >= 2
		disp(sprintf('  primitives..'));
	end
	vhdlbase(dir);
	
	% Generate bit-level FIR filter code
	hdlconf = [];
	if genopts.verbose >= 2
		disp(sprintf('  bit-level code (%s)..', name));
	end
	hdlspec = hdlfir(name, bitspec);
	hdlgen(dir, hdlspec, hdlconf);
	
	% Generate high-level FIR direct form reference code
	firrefname = strcat(name, 'ref');
	if genopts.verbose >= 2
		disp(sprintf('  reference code (%s)..', firrefname));
	end
	refspec = hdlfirref(firrefname, bitspec);
	hdlgen(dir, refspec, hdlconf);
	
	% Generate test bench for bit-level and high-level filters
	firtbname = strcat(name, '_tb');
	if genopts.verbose >= 2
		disp(sprintf('  test bench (%s)..', firtbname));
	end
	tbspec = hdlfirtb(firtbname, hdlspec, refspec);
	hdlgen(dir, tbspec, []);
	
	% Generate stimuli files for testbench
	if genopts.verbose >= 2
		disp(sprintf('  stimuli files..'));
	end
	genfirstim(dir, bitspec, stimlen);
	
	% Generate simulation script
	if genopts.verbose >= 2
		disp(sprintf('  simulation script..'));
	end
	firsrc = hdlsources(hdlspec, hdlconf);
	tbsrc = cat(1, hdlsources(refspec, hdlconf), hdlsources(tbspec, hdlconf));
	genmakefile(dir, name, firsrc, tbsrc);
	
	if genopts.verbose >= 1
		disp(sprintf('Generated VHDL files (bit-level code):'));
		disp(sprintf(' %s', firsrc{:}));
		disp(sprintf('Generated VHDL files (reference code and test bench):'));
		disp(sprintf(' %s', tbsrc{:}));
	end
	
